package at.kugel.tool.buildtray.status;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Label;
import java.awt.SystemTray;
import java.awt.TextArea;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

/**
 * Simulate a system tray popup which closes on clicked. Used because older versions of Java 6 do not provide this in Windows systems.
 *
 * @author <a href="http://www.code-cop.org/">Peter Kofler</a>
 */
class TrayPopup extends Frame {

   private static final Color FG_COLOR = Color.BLACK;
   private static final Color BG_COLOR = new Color(255, 255, 220);
   private static final String FONT_NAME = "Arial";

   private final Label title = new Label();
   private final TextArea message = new TextArea("", 1, 1, TextArea.SCROLLBARS_NONE);

   public TrayPopup() {
      setWindowProperties();
      setComponentProperties(this);
      setLayout(new BorderLayout());
      addTitleLabel();
      addMessageArea();
   }

   private void setWindowProperties() {
      setUndecorated(true);
      setResizable(false);
      setAlwaysOnTop(true);
      setFocusableWindowState(false);
   }

   private void addTitleLabel() {
      setComponentProperties(title);

      title.setAlignment(Label.CENTER);
      title.setFont(new Font(FONT_NAME, Font.BOLD, 12));

      add(title, BorderLayout.NORTH);
   }

   private void addMessageArea() {
      setComponentProperties(message);

      message.setEditable(false);

      add(message, BorderLayout.CENTER);
   }

   private void close() {
      setVisible(false);
      dispose();
   }

   private void setComponentProperties(Component component) {
      component.setFocusable(false);
      component.addMouseListener(new MouseAdapter() {
         @Override
         public void mouseClicked(MouseEvent pE) {
            close();
         }
      });
      component.setBackground(BG_COLOR);
      component.setForeground(FG_COLOR);
      component.setFont(new Font(FONT_NAME, Font.PLAIN, 10));
   }

   void displayMessage(String caption, String text) {
      updateTitle(caption);
      updateTextArea(text);

      pack();
      center();
      setVisible(true);
   }

   private void updateTitle(String caption) {
      title.setText(caption);
   }

   private void updateTextArea(String text) {
      String[] lines = text.split("\n");
      setTextAreaRows(lines);
      setTextAreaColumns(lines);
      message.setText(text);
   }

   private void setTextAreaRows(String[] lines) {
      message.setRows(lines.length);
   }

   private void setTextAreaColumns(String[] lines) {
      message.setColumns(1);
      for (String l : lines) {
         if (l.length() > message.getColumns()) {
            message.setColumns(l.length());
         }
      }
   }

   private void center() {
      int mostRight = calcMostRightPosition();
      int justAboveSystemTray = calcMostDownPosition();
      setLocation(mostRight, justAboveSystemTray);
   }

   private int calcMostRightPosition() {
      Dimension mySize = getSize();
      Dimension screenSize = getToolkit().getScreenSize();
      return screenSize.width - mySize.width;
   }

   private int calcMostDownPosition() {
      Dimension mySize = getSize();
      Dimension screenSize = getToolkit().getScreenSize();
      int trayHeight = SystemTray.getSystemTray().getTrayIconSize().height;
      final int niceDistance = 4;

      return screenSize.height - mySize.height - trayHeight - niceDistance;
   }

}
