package at.kugel.tool.buildtray.status;

import java.awt.Image;
import java.awt.TrayIcon;
import java.awt.TrayIcon.MessageType;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.EnumMap;

/**
 * Provides methods to change the tray icon and popup messages. This just wrapps around the tray iconto be SetStatusAble.
 *
 * @author <a href="http://www.code-cop.org/">Peter Kofler</a>
 */
public class TrayIconStatus implements SetStatusAble {

   private final EnumMap<Status, Image> images;
   private final TrayIcon trayIcon;

   private Status previousStatus;
   private String previousProject;
   private Status currentStatus = Status.BUSY;
   private String currentProject;
   private String buildDate;

   public TrayIconStatus(TrayIcon trayIcon, EnumMap<Status, Image> images) {
      this.trayIcon = trayIcon;
      this.images = images;
   }

   /**
    * Some time later the application state has changed.
    */
   @Override
   public void setStatus(Status newStatus, String newProject, String date) {
      savePreviousStatus();

      currentStatus = newStatus;
      currentProject = newProject;
      buildDate = date;

      updateImage();
      updateBalloon();
   }

   private void savePreviousStatus() {
      previousStatus = currentStatus;
      previousProject = currentProject;
   }

   private void updateImage() {
      trayIcon.setImage(images.get(currentStatus));
   }

   private void updateBalloon() {
      switch (currentStatus) {
      case OK:
         updateBalloonOk();
         break;
      case FAILED:
         updateBalloonFailed();
         break;
      case BUSY:
         updateBalloonBusy();
         break;
      default:
         throw new IllegalArgumentException("impossible unknown status value " + currentStatus);
      }
   }

   private void updateBalloonOk() {
      String message = "all are " + currentStatus + ' ' + buildDate;
      trayIcon.setToolTip(message);
   }

   private void updateBalloonFailed() {
      String message = currentProject + " is " + currentStatus + ' ' + buildDate;
      trayIcon.setToolTip(message);

      if (sameProjectChangedToFailed()) {
         displayMessage("Build Failed", message, TrayIcon.MessageType.NONE);
      }
   }

   private boolean sameProjectChangedToFailed() {
      return currentStatus.isWorseThan(previousStatus) && currentProject.equals(previousProject);
   }

   private void updateBalloonBusy() {
      String message = currentProject + " is " + currentStatus;
      trayIcon.setToolTip(message);

      if (workingOnDifferentProject()) {
         displayMessage("New Build", "Finished " + previousProject + "\nBuilding " + currentProject, TrayIcon.MessageType.NONE);
      }
   }

   private boolean workingOnDifferentProject() {
      return previousProject != null && !currentProject.equals(previousProject);
   }

   @Override
   public void setError(Exception ex) {
      String trace = printTrace(ex);
      displayMessage("Error " + ex.getMessage(), trace, TrayIcon.MessageType.ERROR);
   }

   private String printTrace(Exception ex) {
      StringWriter out = new StringWriter();
      ex.printStackTrace(new PrintWriter(out));
      return out.toString().replaceAll("\\s+", " ");
   }

   /**
    * Wrapper for the tray icon's popup, which does not work on windows XP (known bug).
    *
    * @see TrayIcon#displayMessage
    */
   private void displayMessage(String caption, String text, MessageType messageType) {
      if ("5.1".equals(System.getProperty("os.version"))) {
         new TrayPopup().displayMessage(caption, text);
      } else {
         trayIcon.displayMessage(caption, text, messageType);
      }
   }

}
