package at.kugel.tool.buildtray.config;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Properties;
import java.util.regex.Pattern;

/**
 * Config tracks file date and reloads itself.
 *
 * @author <a href="http://www.code-cop.org/">Peter Kofler</a>
 */
public class ReloadingConfig implements Config {

   private final File path;
   private long lastLoaded;
   private Properties props;

   public ReloadingConfig(String configPath) {
      if (configPath == null) {
         throw new IllegalArgumentException("path is null");
      }
      path = new File(configPath);
   }

   @Override
   public File getConfigFileLocation() {
      return path;
   }

   private void checkForUpdate() {
      final long lastModified = configLastModified();
      if (lastLoaded < lastModified) {
         reloadPropertiesHandleErrors();
         lastLoaded = lastModified;
      }
   }

   private long configLastModified() {
      return path.lastModified();
   }

   private void reloadPropertiesHandleErrors() {
      try {
         reloadProperties();
      } catch (IOException e) {
         throw new IllegalStateException(e);
      }
   }

   private void reloadProperties() throws IOException {
      props = new Properties();
      final Reader in = new FileReader(path);
      try {
         props.load(in);
      } finally {
         in.close();
      }
   }

   @Override
   public long getPollingMs() {
      return getInteger("polling.ms");
   }

   private int getInteger(String key) {
      return Integer.parseInt(getValue(key));
   }

   private String getValue(String key) {
      checkForUpdate();
      return props.getProperty(key);
   }

   private URL getUrl(String key) {
      try {

         String value = getValue(key);
         if (value != null && !value.equals("")) {
            return new URL(value);
         }
         return null;
         
      } catch (MalformedURLException e) {
         throw new IllegalArgumentException("broken URL in config", e);
      }
   }
   
   @Override
   public URL getServerlUrl() {
      return getUrl("server.url");
   }

   @Override
   public URL getServerDisplayUrl() {
      return getUrl("server.display");
   }

   @Override
   public CheckBusy getCheckBusy() {
      return CheckBusy.fromConfig(getValue("busy.check"));
   }

   private boolean getBoolean(String key) {
      return Boolean.valueOf(getValue(key));
   }

   @Override
   public DateFormat getDatePattern() {
      return new SimpleDateFormat(getValue("status.date.format"));
   }

   @Override
   public long getSplashWaitMs() {
      return getInteger("splash.wait.ms");
   }

   @Override
   public Pattern getBusyPattern() {
      String busyPattern = getValue("busy.pattern");
      return Pattern.compile(busyPattern);
   }

   @Override
   public String getSuccessValue() {
      return getValue("value.success");
   }

   @Override
   public String getFailedValue() {
      return getValue("value.failed");
   }

   @Override
   public Pattern getStatusPattern() {
      String statusPattern = getValue("status.pattern");
      return Pattern.compile(statusPattern, Pattern.DOTALL);
   }

   @Override
   public int getStatusPatternNameGroup() {
      return getInteger("status.name.group");
   }

   @Override
   public int getStatusPatternStatusGroup() {
      return getInteger("status.value.group");
   }

   @Override
   public int getStatusPatternDateGroup() {
      return getInteger("status.date.group");
   }

   @Override
   public String getProxyHost() {
      return getValue("proxy.host");
   }

   @Override
   public int getProxyPort() {
      return getInteger("proxy.port");
   }

   @Override
   public boolean useProxy() {
      return getBoolean("proxy.set");
   }

   @Override
   public boolean useAuth() {
      return getBoolean("server.basic.auth");
   }

   @Override
   public String getServerUsername() {
      return getValue("server.username");
   }

   @Override
   public String getServerPassword() {
      return getValue("server.password");
   }

   @Override
   public String getTrustStore() {
      return getValue("server.trust.store.location");
   }

   @Override
   public String getTrustStorePassword() {
      return getValue("server.trust.store.password");
   }

   @Override
   public boolean useTrustStore() {
      return getBoolean("server.trust.store");
   }

}
