package at.kugel.tool.buildtray.action;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import at.kugel.tool.buildtray.config.Config;
import at.kugel.tool.buildtray.util.Base64;
import at.kugel.tool.buildtray.util.HttpHeaderEntry;
import at.kugel.tool.buildtray.util.HttpHeaders;

/**
 * Strategy for String related HTTP methods.
 *
 * @author <a href="http://www.code-cop.org/">Peter Kofler</a>
 */
public class URLReader {

   private final Config config;
   /**
    * The headers to send and on exit the headers which were sent.
    */
   private final List<Map.Entry<String, String>> headers = new ArrayList<Map.Entry<String, String>>();

   public URLReader(Config networkConfig) {
      config = networkConfig;

      setProxySettings();
      setSSLSettings();
   }

   private void setProxySettings() {
      if (config.useProxy()) {
         setProxy();
      } else {
         setNoProxy();
      }
   }

   private void setProxy() {
      System.setProperty("proxySet", Boolean.TRUE.toString());
      System.setProperty("proxyHost", config.getProxyHost());
      System.setProperty("proxyPort", Integer.toString(config.getProxyPort()));
   }

   private void setNoProxy() {
      System.setProperty("proxySet", Boolean.FALSE.toString());
      System.getProperties().remove("proxyHost");
      System.getProperties().remove("proxyPort");
   }

   private void setSSLSettings() {
      if (config.getServerlUrl().getProtocol().equals("https")) {
         if (config.useTrustStore()) {
            setTrustStore();
         } else {
            disableHttpsStores();
         }
      }
   }

   private void setTrustStore() {
      System.setProperty("javax.net.ssl.trustStore", config.getTrustStore());
      System.setProperty("javax.net.ssl.trustStorePassword", config.getTrustStorePassword());
   }

   private void disableHttpsStores() {
      System.getProperties().remove("javax.net.ssl.trustStore");
      System.getProperties().remove("javax.net.ssl.trustStorePassword");
      SSLHelper.disableHTTPSChecks();
   }

   public String readPage() throws IOException {
      setBasicAuth();
      return readNet();
   }

   private void setBasicAuth() throws UnsupportedEncodingException {
      if (config.useAuth()) {
         String userPassword = config.getServerUsername() + ':' + config.getServerPassword();
         String encoding = Base64.encode(userPassword.getBytes(), "8859_1");
         headers.add(new HttpHeaderEntry("Authorization", "Basic " + encoding));
      }
   }

   /**
    * Load a textfile from the net into a string.
    *
    * @return a string of the file loaded.
    * @throws IOException if the string can not be read.
    */
   private String readNet() throws IOException {
      URLConnection con = openConnection();

      int size = con.getContentLength();
      boolean sizeGiven = true;
      if (size < 0) {
         sizeGiven = false;
         // guess size
         size = 256 * 1024;
      }

      final byte[] data = new byte[size];

      InputStream ins = null;
      try {
         ins = con.getInputStream();
         // comes in parts - read parts
         int pos = 0;
         int res = 1;
         while (pos < size && res > 0) {
            res = ins.read(data, pos, size - pos);
            pos += res;
         }

         if (sizeGiven && pos < size) {
            throw new IOException("short file: length=" + pos + " != content-length=" + size);
         } else if (!sizeGiven && pos == size) {
            throw new IOException("possibly long file: length=" + pos + " != content-length=unknown");
         } else if (!sizeGiven) {
            pos++;
         }

         final String encoding = con.getContentEncoding();
         if (encoding == null) {
            return new String(data, 0, pos);
         }

         return new String(data, 0, pos, encoding);

      } finally {
         if (ins != null) {
            ins.close();
         }
      }

   }

   private URLConnection openConnection() throws IOException {
      URL url = config.getServerlUrl();
      URLConnection con = url.openConnection();
      HttpHeaders.setInto(con, headers);
      con.connect();
      HttpHeaders.readFrom(con, headers);
      return con;
   }
}
