package at.kugel.tool.buildtray.action;

import java.awt.Desktop;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.StringReader;
import java.io.StringWriter;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import at.kugel.tool.buildtray.config.Config;
import at.kugel.tool.buildtray.status.SetStatusAble;
import at.kugel.tool.buildtray.util.DesktopOperation;

/**
 * Download the latest output from the build server and display it. This is helpfull during setup.Save it to a file and invoke Desktop EDIT.
 *
 * @author <a href="http://www.code-cop.org/">Peter Kofler</a>
 */
class ShowServerResponseCommand extends ActionCommand {

   public ShowServerResponseCommand(Config config, SetStatusAble statusDisplay) {
      super(config, statusDisplay);
   }

   @Override
   protected void workTemplate() throws IOException, TransformerException {
      showServerResponse();
   }

   private void showServerResponse() throws IOException, TransformerException {
      String raw = downloadBuildStatus();
      String formatted = format(raw);
      String annotated = annotate(formatted);
      File savedFile = saveToFile(annotated);
      openDesktopEditor(savedFile);
   }

   private String downloadBuildStatus() throws IOException {
      return new URLReader(config).readPage();
   }

   private String format(String content) throws TransformerException {
      if (isXml(content)) {
         return formatXml(content);
      }
      return content;
   }

   private boolean isXml(String content) {
      return content.startsWith("<?xml");
   }

   private String formatXml(String content) throws TransformerException {
      // see http://stackoverflow.com/questions/139076/how-to-pretty-print-xml-from-java
      // see http://johnsonsolutions.blogspot.com/2007/08/xml-transformer-indent-doesnt-work-with.html
      TransformerFactory transFactory = TransformerFactory.newInstance();
      transFactory.setAttribute("indent-number", Integer.valueOf(4));
      Transformer transformer = transFactory.newTransformer();
      transformer.setOutputProperty(OutputKeys.INDENT, "yes");

      Source source = new StreamSource(new StringReader(content));
      StreamResult result = new StreamResult(new StringWriter());
      transformer.transform(source, result);
      return result.getWriter().toString();
   }

   private String annotate(String content) {
      RegexBusyParser busyParser = new RegexBusyParser(config, content);
      if (busyParser.isBusy()) {
         return busyParser.annotate();
      }

      RegexStatusParser statusParser = new RegexStatusParser(config, content);
      return statusParser.annotate();
   }

   private File saveToFile(String content) throws IOException {
      File tmpFile = createTempFile();
      save(content, tmpFile);
      return tmpFile;
   }

   private File createTempFile() throws IOException {
      File tmpFile = File.createTempFile("build-server-tray-log-", ".txt");
      tmpFile.deleteOnExit();
      return tmpFile;
   }

   private void save(String content, File tmpFile) throws IOException {
      OutputStream out = new FileOutputStream(tmpFile);
      try {
         out.write(content.getBytes());
         out.flush();
      } finally {
         out.close();
      }
   }

   private void openDesktopEditor(final File savedFile) throws IOException {
      new DesktopOperation(Desktop.Action.EDIT) {
         @Override
         public void runWithDesktop(Desktop deskTop) throws IOException {
            deskTop.edit(savedFile);
         }
      }.run();
   }

}
