package at.kugel.tool.buildtray;

import java.awt.AWTException;
import java.awt.SplashScreen;
import java.awt.SystemTray;
import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;

import at.kugel.tool.buildtray.action.Actions;
import at.kugel.tool.buildtray.action.HtmlRegularExpressionActions;
import at.kugel.tool.buildtray.config.Config;
import at.kugel.tool.buildtray.config.ReloadingConfig;
import at.kugel.tool.buildtray.status.SetStatusAble;

/**
 * Build Tray Main Starter. Loads images display the tray, start polling threads and close splash screen.
 *
 * @author <a href="http://www.code-cop.org/">Peter Kofler</a>
 */
public final class Main {

   public static void main(String[] args) throws AWTException, InterruptedException {
      handleTrayNotSupported();

      handleEmptyArguments(args);

      String configFileName = args[0];
      Config conf = new ReloadingConfig(configFileName);

      Actions actions = new HtmlRegularExpressionActions(conf);
      SetStatusAble tray = new TrayIconFactory().create(actions);
      actions.setStatusDisplay(tray);

      setUpRefreshTimer(conf.getPollingMs(), actions);

      closeSplashScreen(conf.getSplashWaitMs());
   }

   private static void handleTrayNotSupported() {
      if (!SystemTray.isSupported()) {
         System.err.println("System Tray not supported");
         System.exit(1);
      }
   }

   private static void handleEmptyArguments(String[] args) {
      if (args.length != 1) {
         System.out.println("Usage: java -jar BuildServerSystemTray.jar <path to config file, e.g. \"config/Hudson-1.2.properties\">");
         System.exit(2);
      }
   }

   private static void setUpRefreshTimer(long pollingMs, final Actions actions) {
      Timer timer = new Timer("Build Server Polling", true);
      timer.scheduleAtFixedRate(new TimerTask() {
         @Override
         public void run() {
            actions.refresh();
         }
      }, new Date(), pollingMs);
   }

   private static void closeSplashScreen(long splashWaitMs) throws InterruptedException {
      SplashScreen screen = SplashScreen.getSplashScreen();
      if (screen != null) {
         Thread.sleep(splashWaitMs);
         screen.close();
      }
   }

}
